//
// (c) 2020 wesolutions GmbH
// All rights reserved.
//

import QtQuick 2.3

import wesual.Controls 1.0

import wesual.data.Core        1.0
import wesual.data.FoodService 1.0

PopupAnchorItem {
    id : flyout

    property Context context

    signal productPicked(FoodServiceProduct product)

    anchors.centerIn : parent

    popup : Rectangle {
        id : menu

        function resetFilter() {
            filteredName.text = "";
            productNameFilter.propertyFilter = "";
            filteredName.forceActiveFocus();

            // stop the timer because it will be restarted when text is changing
            if (timer.running) {
                timer.stop();
            }
        }

        PersistentTypeListing {
            id : productListing

            type : "product:foodservice"
            context : flyout.context
            queryLimit : 100
        }

        DocumentListFilter {
            id : filteredProductListing
            documents : productListing.documents
            filters : productNameFilter
        }

        DocumentListSorter {
            id : productSortedList
            documents : filteredProductListing.resultingDocuments
            sorters : TypeSorter {
                property : "name"
            }
        }

        Keys.onPressed : {
            if (event.key === Qt.Key_PageUp || event.key === Qt.Key_PageDown) {
                // do not propagate navigation keys while focus item
                event.accepted = true;
            }
        }
        Keys.onRightPressed : {}
        Keys.onLeftPressed : {}
        Keys.onUpPressed   : {
            productView.currentIndex = Math.max(0, productView.currentIndex - 1)
        }

        Keys.onDownPressed : {
            productView.currentIndex =
                    Math.min(productView.currentIndex + 1,
                             productSortedList.resultingDocuments.length - 1);
        }

        StringFilter {
            id : productNameFilter
            property : "name"
        }

        readonly property int preferredHeight :
            productView.contentHeight + filteredName.height + 15
        readonly property bool flipped : flyout.distanceBottom < 400
        readonly property bool shift   : flyout.distanceRight < width
        property int  currentIndex : 0

        width : 302
        height : Math.min(
                     preferredHeight,
                     (flipped ? flyout.distanceTop : flyout.distanceBottom)
                     - 40)

        color : UI.color(UI.PrimaryBase)
        border {
            width : 1
            color : UI.color(UI.PrimaryControlBorder)
        }

        opacity : 0
        onOpacityChanged : {
            if (opacity === 0)
                menu.resetFilter();
        }

        anchors {
            left : parent.left
            leftMargin : {
                if (shift) {
                    var leftMargin = - flyout.parent.width / 2;
                    var shiftTerm  =
                            Math.max(-width, -(width - flyout.distanceRight));
                    return leftMargin += shiftTerm;
                } else
                    return -flyout.parent.width / 2;
            }
            top : flipped ? undefined : parent.bottom
            topMargin : flyout.parent.height / 2
            bottom : flipped ? parent.top : undefined
            bottomMargin : flyout.parent.height / 2
        }

        states : [
            State {
                name : "visible"
                when : flyout.popupVisible

                PropertyChanges {
                    target : menu
                    opacity : 1
                }
            }
        ]

        transitions : Transition {
            NumberAnimation {
                property : "opacity"
                duration : 200
            }
        }

        TextField {
            id : filteredName

            // force focus on reopen
            Connections {
                target : flyout
                onPopupVisibleChanged : {
                    if(popupVisible)
                        filteredName.forceActiveFocus();
                }
                ignoreUnknownSignals : true
            }
            // force focus on Open
            Component.onCompleted :  {
                menu.resetFilter();
            }

            anchors {
                top : parent.top
                topMargin : 5
                left : parent.left
                leftMargin : 5
                right : parent.right
                rightMargin : 5
            }

            onInputAccepted : {
                var chosenProduct =
                    productSortedList.resultingDocuments[productView.currentIndex];

                if (chosenProduct) {
                    flyout.productPicked(chosenProduct);
                }
                flyout.hide();
            }

            height : 30

            Timer {
                id : timer
                interval : 500
                running : false
                repeat : false

                onTriggered :
                    productNameFilter.propertyFilter = filteredName.text
            }

            onTextChanged : timer.restart()
        }

        ScrollView {

            anchors {
                left : parent.left
                leftMargin : 2
                rightMargin : 2
                right : parent.right
                top : filteredName.bottom
                topMargin : 4
                bottom : parent.bottom
                bottomMargin : 5
            }

            ListView {
                id : productView

                model : productSortedList.resultingDocuments

                delegate : Item {
                    width : parent ? parent.width : 100
                    height : 68

                    Rectangle {
                        id : selectionRect

                        anchors {
                            left : parent.left
                            leftMargin : 3
                            right : parent.right
                            rightMargin : 3
                            top : parent.top
                            bottom : parent.bottom
                        }

                        color : productView.currentIndex === index
                                ? UI.color(UI.PrimaryHover)
                                  : UI.color(UI.PrimaryBase)

                        Image {
                            id : image

                            asynchronous : true
                            width : 60
                            height : width
                            mipmap : true
                            smooth : true
                            fillMode : Image.PreserveAspectCrop
                            source : modelData.primaryImage ?
                                     modelData.primaryImage.thumbnail :
                                     ""
                            anchors {
                                left : parent.left
                                leftMargin : 6
                                verticalCenter : parent.verticalCenter
                            }
                        }

                        Column {
                            id : textColumn

                            anchors {
                                left : image.right
                                leftMargin : 9
                                right : parent.right
                                top : parent.top
                                topMargin : 5
                                //verticalCenter: parent.verticalCenter
                            }

                            Text {
                                anchors {
                                    left : parent.left
                                    right : parent.right
                                }

                                text : modelData.name
                                font {
                                    family : UI.fontFamily(UI.PrimaryFont)
                                    pixelSize : 14
                                }

                                maximumLineCount: 2
                                wrapMode : Text.Wrap
                                color : UI.color(UI.PrimaryTextColor)
                            }
                            PriceLabel {
                                font {
                                    family : UI.fontFamily(UI.SecondaryFont)
                                    pixelSize : 12
                                }
                                product : modelData
                                lineHeightMode :  Text.FixedHeight
                                lineHeight : 26
                                color : UI.color(UI.PrimaryTextColor)
                            }
                        }

                        MouseArea {
                            id : selectionMouseArea

                            anchors.fill : parent
                            hoverEnabled: true
                            onContainsMouseChanged : {
                                if (containsMouse)
                                    productView.currentIndex = index;
                            }

                            onClicked : {
                                modelData.beginRecordChanges();
                                flyout.productPicked(modelData);
                                modelData.endRecordChanges();
                                flyout.hide();
                            }
                        }
                    }
                }
            }
        }
    }
}
